﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace VA.PPMS.IWS.Common
{
    public class PayloadProcessingSetting
    {
        const int DefaultOffset = 5;
        const int DefaultWait = 15;

        public int WaitTime { get; set; }

        public int? StartHour { get; set; }

        public int? EndHour { get; set; }

        public int? UtcStart => StartHour.HasValue ? ConvertHourToUtc(StartHour.Value) : (int?)null;

        public int? UtcEnd => EndHour.HasValue ? ConvertHourToUtc(EndHour.Value) : (int?)null;

        public int Offset { get; set; }

        public bool IsValid => StartHour.HasValue && EndHour.HasValue;

        public PayloadProcessingSetting()
        {
            WaitTime = DefaultWait;
        }

        public PayloadProcessingSetting(int start, int end, int offset = DefaultOffset, int waitTime = DefaultWait)
        {
            StartHour = start;
            EndHour = end;
            Offset = offset;
            SetWaitTime(waitTime);
        }

        public PayloadProcessingSetting(string start, string end, int offset = DefaultOffset, int waitTime = DefaultWait)
        {
            Init(start, end, offset, waitTime);
        }

        public PayloadProcessingSetting(string settings)
        {
            if (string.IsNullOrWhiteSpace(settings)) return;

            string[] paramList = settings.Split(',');

            int? offset = null;
            int? wait = null;

            if (paramList.Length > 3)
            {
                offset = int.TryParse(paramList[3], out int offsetTarget) ? offsetTarget : (int?)null;
                wait = int.TryParse(paramList[2], out int waitTarget) ? waitTarget : (int?)null;
            }
            else if (paramList.Length > 2)
            {
                wait = int.TryParse(paramList[2], out int waitTarget) ? waitTarget : (int?)null;
            }

            Init(paramList[0], paramList[1], offset, wait);
        }

        private void Init(string start, string end, int? offset = DefaultOffset, int? waitTime = DefaultWait)
        {
            SetStartHour(start);
            SetEndHour(end);
            Offset = offset ?? DefaultOffset;
            SetWaitTime(waitTime);
        }

        private void SetStartHour(string start)
        {
            StartHour = int.TryParse(start, out int target) ? target : (int?)null;
        }

        private void SetEndHour(string end)
        {
            EndHour = int.TryParse(end, out int target) ? target : (int?)null;
        }

        private void SetWaitTime(int? waitTime)
        {
            WaitTime = waitTime.HasValue && waitTime.Value > 0 && waitTime.Value < 24 ? waitTime.Value : DefaultWait;
        }

        private int ConvertHourToUtc(int hour)
        {
            var ts = new TimeSpan(hour, 0, 0);
            ts = ts.Add(new TimeSpan(Offset, 0, 0));
            return ts.Hours;
        }
    }
}
